package com.foobar.dragbug;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.ICoolBarManager;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.ToolBarContributionItem;
import org.eclipse.jface.action.ToolBarManager;
import org.eclipse.swt.SWT;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.ActionFactory.IWorkbenchAction;
import org.eclipse.ui.application.ActionBarAdvisor;
import org.eclipse.ui.application.IActionBarConfigurer;

/**
 * An action bar advisor is responsible for creating, adding, and disposing of the
 * actions added to a workbench window. Each window will be populated with
 * new actions.
 */
public class ApplicationActionBarAdvisor extends ActionBarAdvisor {

	// Actions - important to allocate these only in makeActions, and then use them
	// in the fill methods. This ensures that the actions aren't recreated
	// when fillActionBars is called with FILL_PROXY.
	private IWorkbenchAction exitAction;
	private IWorkbenchAction aboutAction;
	private IWorkbenchAction newWindowAction;
	private OpenEditorAction openViewAction;
	private Action messagePopupAction;

	private Action saveWorkbenchAction;
	private Action restoreWorkbenchAction;

	public ApplicationActionBarAdvisor(IActionBarConfigurer configurer) {
		super(configurer);
	}

	@Override
	protected void makeActions(final IWorkbenchWindow window) {
		// Creates the actions and registers them.
		// Registering is needed to ensure that key bindings work.
		// The corresponding commands keybindings are defined in the plugin.xml file.
		// Registering also provides automatic disposal of the actions when
		// the window is closed.

		this.exitAction = ActionFactory.QUIT.create(window);
		register(this.exitAction);

		this.aboutAction = ActionFactory.ABOUT.create(window);
		register(this.aboutAction);

		this.newWindowAction = ActionFactory.OPEN_NEW_WINDOW.create(window);
		register(this.newWindowAction);

		this.openViewAction = new OpenEditorAction(window, "Open Another Message View", Editor.ID);
		register(this.openViewAction);

		this.messagePopupAction = new MessagePopupAction("Open Message", window);
		register(this.messagePopupAction);

		this.saveWorkbenchAction = new Action("Save") {
			@Override
			public void run() {
				WorkbenchMemory.memorize();
			}
		};
		this.saveWorkbenchAction.setId("saveWorkbenchAction");
		register(this.saveWorkbenchAction);

		this.restoreWorkbenchAction = new Action("Restore") {
			@Override
			public void run() {
				WorkbenchMemory.remember();
			}
		};
		this.restoreWorkbenchAction.setId("restoreWorkbenchAction");
		register(this.restoreWorkbenchAction);
	}

	@Override
	protected void fillMenuBar(IMenuManager menuBar) {
		MenuManager fileMenu = new MenuManager("&File", IWorkbenchActionConstants.M_FILE);
		MenuManager helpMenu = new MenuManager("&Help", IWorkbenchActionConstants.M_HELP);
		MenuManager bugMenu = new MenuManager("&Bug", IWorkbenchActionConstants.M_HELP);

		menuBar.add(fileMenu);
		menuBar.add(bugMenu);
		// Add a group marker indicating where action set menus will appear.
		menuBar.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
		menuBar.add(helpMenu);

		// File
		fileMenu.add(this.newWindowAction);
		fileMenu.add(new Separator());
		fileMenu.add(this.messagePopupAction);
		fileMenu.add(this.openViewAction);
		fileMenu.add(new Separator());
		fileMenu.add(this.exitAction);

		// Bug
		bugMenu.add(this.saveWorkbenchAction);
		bugMenu.add(this.restoreWorkbenchAction);

		// Help
		helpMenu.add(this.aboutAction);
	}

	@Override
	protected void fillCoolBar(ICoolBarManager coolBar) {
		IToolBarManager toolbar = new ToolBarManager(SWT.FLAT | SWT.RIGHT);
		coolBar.add(new ToolBarContributionItem(toolbar, "main"));
		toolbar.add(this.openViewAction);
		toolbar.add(this.messagePopupAction);
	}
}
